<?php

namespace GenixCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Apexa Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class SERVICES_TAB extends Widget_Base
{

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'services-tab';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Services Tab', 'genixcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'genix-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['genixcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['genixcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            '__tg_services_section',
            [
                'label' => esc_html__('Services Tabs', 'genixcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->start_controls_tabs(
            '_tab_style_top_box'
        );
        $repeater->start_controls_tab(
            '_tab_left_content',
            [
                'label' => esc_html__('Tab Button', 'genixcore'),
            ]
        );

        if (genix_is_elementor_version('<', '2.6.0')) {
            $repeater->add_control(
                'tg_icon',
                [
                    'label' => esc_html__('Upload Icon', 'genixcore'),
                    'show_label' => false,
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'fa fa-star',
                ]
            );
        } else {
            $repeater->add_control(
                'tg_selected_icon',
                [
                    'label' => esc_html__('Upload Icon', 'genixcore'),
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'label_block' => true,
                    'default' => [
                        'value' => 'flaticon-healthcare',
                        'library' => 'solid',
                    ],
                ]
            );
        }

        $repeater->add_control(
            'tab_text',
            [
                'label' => esc_html__('Button Text', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Health Insurance', 'genixcore'),
                'label_block' => true,
            ]
        );

        $repeater->end_controls_tab();

        $repeater->start_controls_tab(
            '_tab_right_content',
            [
                'label' => esc_html__('Tab Content', 'genixcore'),
            ]
        );

        $repeater->add_control(
            'tg_services_image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => esc_html__('Image', 'genixcore'),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'tab_title',
            [
                'label' => esc_html__('Title', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Health Insurance', 'genixcore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'tab_desc',
            [
                'label' => esc_html__('Description', 'genixcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Explore our savings, quality care and wellness solutions to craft the perfect plan for your busineshen an unknown printer.', 'genixcore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'tab_html',
            [
                'label' => esc_html__('Item List HTML', 'genixcore'),
                'type' => Controls_Manager::CODE,
                'language' => 'html',
                'rows' => 20,
            ]
        );

        $repeater->add_control(
            'btn_text',
            [
                'label' => esc_html__('Button Text', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Read More', 'genixcore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'tg_services_url',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => esc_html__('Button URL', 'genixcore'),
                'default' => esc_html__('#', 'genixcore'),
                'placeholder' => esc_html__('Type url here', 'genixcore'),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->end_controls_tab();
        $repeater->end_controls_tabs();

        $this->add_control(
            'tg_services_lists',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => esc_html__('Services Item', 'genixcore'),
                'default' => [
                    [
                        [
                            'tab_text' => esc_html__('Health Insurance', 'genixcore'),
                        ],
                        [
                            'tab_text' => esc_html__('Travel Insurance', 'genixcore'),
                        ],
                        [
                            'tab_text' => esc_html__('Vehicle Insurance', 'genixcore'),
                        ],
                    ],
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'default' => 'medium_large',
                'separator' => 'before',
                'exclude' => [
                    'custom'
                ]
            ]
        );

        $this->end_controls_section();


        // STYLE TAB
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'genixcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'text_transform',
            [
                'label' => esc_html__('Text Transform', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => esc_html__('None', 'genixcore'),
                    'uppercase' => esc_html__('UPPERCASE', 'genixcore'),
                    'lowercase' => esc_html__('lowercase', 'genixcore'),
                    'capitalize' => esc_html__('Capitalize', 'genixcore'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display(); ?>


        <div class="services__tab-wrap">
            <ul class="nav nav-tabs" id="myTab" role="tablist">
                <?php foreach ($settings['tg_services_lists'] as $key => $tab) :
                    $active = ($key == 0) ? 'active' : '';
                ?>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link <?php echo esc_attr($active); ?>" id="apexa-tab-<?php echo esc_attr($key); ?>" data-bs-toggle="tab" data-bs-target="#apexa-<?php echo esc_attr($key); ?>" type="button" role="tab" aria-controls="apexa-<?php echo esc_attr($key); ?>" aria-selected="true">
                            <?php if (!empty($tab['tg_icon']) || !empty($tab['tg_selected_icon']['value'])) : ?>
                                <?php genix_render_icon($tab, 'tg_icon', 'tg_selected_icon'); ?>
                            <?php endif; ?>
                            <?php echo esc_html($tab['tab_text']); ?>
                        </button>
                    </li>
                <?php endforeach; ?>
            </ul>
            <div class="tab-content" id="myTabContent">
                <?php foreach ($settings['tg_services_lists'] as $key => $tab) :
                    $active = ($key == 0) ? 'show active' : '';
                ?>
                    <div class="tab-pane fade <?php echo esc_attr($active); ?>" id="apexa-<?php echo esc_attr($key); ?>" role="tabpanel" aria-labelledby="apexa-tab-<?php echo esc_attr($key); ?>" tabindex="0">
                        <div class="services__item-four shine-animate-item">
                            <?php if (!empty($tab['tg_services_image']['url'])) : ?>
                                <div class="services__thumb-four shine-animate">
                                    <img src="<?php echo esc_url($tab['tg_services_image']['url']); ?>" alt="<?php echo esc_attr__('Image', 'Apexa') ?>">
                                </div>
                            <?php endif; ?>
                            <div class="services__content-four">
                                <?php if (!empty($tab['tab_title'])) : ?>
                                    <h2 class="title"><a href="<?php echo esc_url($tab['tg_services_url']) ?>"><?php echo genix_kses($tab['tab_title']) ?></a></h2>
                                <?php endif; ?>
                                <?php if (!empty($tab['tab_desc'])) : ?>
                                    <p><?php echo genix_kses($tab['tab_desc']) ?></p>
                                <?php endif; ?>
                                <?php if (!empty($tab['tab_html'])) : ?>
                                    <div class="about__list-box">
                                        <ul class="list-wrap">
                                            <?php echo $tab['tab_html']; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                                <?php if (!empty($tab['btn_text'])) : ?>
                                    <a href="<?php echo esc_url($tab['tg_services_url']) ?>" class="btn"><?php echo esc_html($tab['btn_text']) ?></a>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>

<?php
    }
}

$widgets_manager->register(new SERVICES_TAB());
