<?php

namespace GenixCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Apexa Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TG_Testimonial extends Widget_Base
{

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'testimonial';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Testimonial', 'genixcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'genix-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['genixcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['genixcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {

        // layout Panel
        $this->start_controls_section(
            'genix_layout',
            [
                'label' => esc_html__('Design Layout', 'genixcore'),
            ]
        );
        $this->add_control(
            'tg_design_style',
            [
                'label' => esc_html__('Select Layout', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'genixcore'),
                    'layout-2' => esc_html__('Layout 2', 'genixcore'),
                    'layout-3' => esc_html__('Layout 3', 'genixcore'),
                    'layout-4' => esc_html__('Layout 4', 'genixcore'),
                    'layout-5' => esc_html__('Layout 5', 'genixcore'),
                    'layout-6' => esc_html__('Layout 6', 'genixcore'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        // Review group
        $this->start_controls_section(
            'review_list',
            [
                'label' => esc_html__('Testimonial List', 'genixcore'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'review_content',
            [
                'label' => esc_html__('Review Content', 'genixcore'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'rows' => 10,
                'default' => esc_html__('“ Morem ipsum dolor sit amet, consectetur adipisc awing elita florai sum dolor sit amet, consectetur area Borem ipsum dolor sit amet, consectetur.”', 'genixcore'),
                'placeholder' => esc_html__('Type your review content here', 'genixcore'),
            ]
        );

        $repeater->add_control(
            'reviewer_image',
            [
                'label' => esc_html__('Reviewer Image', 'genixcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'reviewer_brand',
            [
                'label' => esc_html__('Reviewer Brand', 'genixcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'reviewer_image_size',
                'default' => 'full',
                'exclude' => [
                    'custom'
                ]
            ]
        );

        $repeater->add_control(
            'reviewer_name',
            [
                'label' => esc_html__('Reviewer Name', 'genixcore'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Mr.Robey Alexa', 'genixcore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'reviewer_designation',
            [
                'label' => esc_html__('Reviewer Designation', 'genixcore'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('CEO, Apexa Agency', 'genixcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'reviews_list',
            [
                'label' => esc_html__('Review List', 'genixcore'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' =>  $repeater->get_controls(),
                'default' => [
                    [
                        'reviewer_name' => esc_html__('Mr.Robey Alexa', 'genixcore'),
                    ],
                    [
                        'reviewer_name' => esc_html__('Samuel Peters', 'genixcore'),
                    ],
                    [
                        'reviewer_name' => esc_html__('Robert Fox', 'genixcore'),
                    ],

                ],
                'title_field' => '{{{ reviewer_name }}}',
            ]
        );

        $this->end_controls_section();


        // TAB_STYLE
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'genixcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'text_transform',
            [
                'label' => esc_html__('Text Transform', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => esc_html__('None', 'genixcore'),
                    'uppercase' => esc_html__('UPPERCASE', 'genixcore'),
                    'lowercase' => esc_html__('lowercase', 'genixcore'),
                    'capitalize' => esc_html__('Capitalize', 'genixcore'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'bg_color',
            [
                'label' => esc_html__('Background Color', 'genixcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .testimonial__item-two' => 'background: {{VALUE}}',
                ],
                'condition' => [
                    'tg_design_style' => 'layout-2'
                ]
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

?>

        <?php if ($settings['tg_design_style']  == 'layout-2') : ?>

            <div class="row justify-content-center gutter-24">
                <?php foreach ($settings['reviews_list'] as $item) :
                    if (!empty($item['reviewer_image']['url'])) {
                        $tg_reviewer_image = !empty($item['reviewer_image']['id']) ? wp_get_attachment_image_url($item['reviewer_image']['id'], $item['reviewer_image_size_size']) : $item['reviewer_image']['url'];
                        $tg_reviewer_image_alt = get_post_meta($item["reviewer_image"]["id"], "_wp_attachment_image_alt", true);
                    }
                ?>
                    <div class="col-lg-4 col-md-6">
                        <div class="testimonial__item-two">
                            <?php if (!empty($item['reviewer_image'])) : ?>
                                <div class="testimonial__avatar">
                                    <img src="<?php echo esc_url($tg_reviewer_image); ?>" alt="<?php echo esc_url($tg_reviewer_image_alt); ?>">
                                </div>
                            <?php endif; ?>
                            <div class="testimonial__info-two">
                                <?php if (!empty($item['reviewer_name'])) : ?>
                                    <h2 class="title"><?php echo genix_kses($item['reviewer_name']); ?></h2>
                                <?php endif; ?>
                                <?php if (!empty($item['reviewer_designation'])) : ?>
                                    <span><?php echo genix_kses($item['reviewer_designation']); ?></span>
                                <?php endif; ?>
                            </div>
                            <div class="testimonial__rating">
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                                <i class="fas fa-star"></i>
                            </div>
                            <?php if (!empty($item['review_content'])) : ?>
                                <p><?php echo esc_html($item['review_content']); ?></p>
                            <?php endif; ?>
                            <div class="icon">
                                <img src="<?php echo get_template_directory_uri(); ?>/assets/img/icons/quote.svg" alt="">
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

        <?php elseif ($settings['tg_design_style']  == 'layout-3') : ?>

            <div class="testimonial__item-wrap">
                <div class="swiper-container testimonial-active-two">
                    <div class="swiper-wrapper">
                        <?php foreach ($settings['reviews_list'] as $item) :
                            if (!empty($item['reviewer_image']['url'])) {
                                $tg_reviewer_image = !empty($item['reviewer_image']['id']) ? wp_get_attachment_image_url($item['reviewer_image']['id'], $item['reviewer_image_size_size']) : $item['reviewer_image']['url'];
                                $tg_reviewer_image_alt = get_post_meta($item["reviewer_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                        ?>
                            <div class="swiper-slide">
                                <div class="testimonial__item-three">
                                    <div class="testimonial__rating testimonial__rating-two">
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                    </div>
                                    <?php if (!empty($item['review_content'])) : ?>
                                        <p><?php echo esc_html($item['review_content']); ?></p>
                                    <?php endif; ?>
                                    <div class="testimonial__bottom">
                                        <div class="testimonial__info-three">
                                            <?php if (!empty($item['reviewer_name'])) : ?>
                                                <h4 class="title"><?php echo genix_kses($item['reviewer_name']); ?></h4>
                                            <?php endif; ?>
                                            <?php if (!empty($item['reviewer_designation'])) : ?>
                                                <span><?php echo genix_kses($item['reviewer_designation']); ?></span>
                                            <?php endif; ?>
                                        </div>
                                        <div class="testimonial__icon">
                                            <img src="<?php echo get_template_directory_uri(); ?>/assets/img/icons/quote02.svg" alt="">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <div class="testimonial__nav-two">
                        <div class="testimonial-button-prev"><i class="flaticon-right-arrow"></i></div>
                        <div class="testimonial-button-next"><i class="flaticon-right-arrow"></i></div>
                    </div>
                </div>
            </div>

        <?php elseif ($settings['tg_design_style']  == 'layout-4') : ?>

            <script>
                if (jQuery(document).ready('#rangeValue').length) {
                    function rangeSlide(value) {
                        document.getElementById('rangeValue').innerHTML = value;
                    }
                }
            </script>

            <div class="testimonial__wrap">
                <div class="testimonial__inner-top">
                    <div class="testimonial-slider-dot">
                        <div class="swiper testimonial__nav-three">
                            <div class="swiper-wrapper">
                                <?php foreach ($settings['reviews_list'] as $item) :
                                    if (!empty($item['reviewer_image']['url'])) {
                                        $tg_reviewer_image = !empty($item['reviewer_image']['id']) ? wp_get_attachment_image_url($item['reviewer_image']['id'], $item['reviewer_image_size_size']) : $item['reviewer_image']['url'];
                                        $tg_reviewer_image_alt = get_post_meta($item["reviewer_image"]["id"], "_wp_attachment_image_alt", true);
                                    }
                                ?>
                                    <?php if (!empty($item['reviewer_image'])) : ?>
                                        <div class="swiper-slide">
                                            <button><img src="<?php echo esc_url($tg_reviewer_image); ?>" alt="<?php echo esc_url($tg_reviewer_image_alt); ?>"></button>
                                        </div>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                    <div class="icon"><i class="fas fa-quote-right"></i></div>
                </div>
                <div class="swiper-container testimonial-active-three">
                    <div class="swiper-wrapper">
                        <?php foreach ($settings['reviews_list'] as $item) : ?>
                            <div class="swiper-slide">
                                <div class="testimonial-item testimonial__item-four">
                                    <div class="testimonial-info">
                                        <?php if (!empty($item['reviewer_name'])) : ?>
                                            <h4 class="title"><?php echo genix_kses($item['reviewer_name']); ?></h4>
                                        <?php endif; ?>
                                        <?php if (!empty($item['reviewer_designation'])) : ?>
                                            <span><?php echo genix_kses($item['reviewer_designation']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="testimonial__rating">
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                        <i class="fas fa-star"></i>
                                    </div>
                                    <div class="testimonial-content">
                                        <?php if (!empty($item['review_content'])) : ?>
                                            <p><?php echo esc_html($item['review_content']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <div class="testimonial__nav-four">
                        <div class="testimonial-two-button-prev"><i class="flaticon-right-arrow"></i></div>
                        <div class="testimonial-two-button-next"><i class="flaticon-right-arrow"></i></div>
                    </div>
                </div>
            </div>

        <?php elseif ($settings['tg_design_style']  == 'layout-5') : ?>

            <div class="slider_testimonial_home7">
                <div class="swiper-container slider_baner__active">
                    <div class="swiper-wrapper">
                        <?php foreach ($settings['reviews_list'] as $item) :
                            if (!empty($item['reviewer_image']['url'])) {
                                $tg_reviewer_image = !empty($item['reviewer_image']['id']) ? wp_get_attachment_image_url($item['reviewer_image']['id'], $item['reviewer_image_size_size']) : $item['reviewer_image']['url'];
                                $tg_reviewer_image_alt = get_post_meta($item["reviewer_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                        ?>
                            <div class="swiper-slide slide__home7">
                                <div class="item-testimonial">
                                    <div class="item-testimonial-left">
                                        <div class="author-testimonial">
                                            <?php if (!empty($item['reviewer_image'])) : ?>
                                                <img src="<?php echo esc_url($tg_reviewer_image); ?>" alt="<?php echo esc_url($tg_reviewer_image_alt); ?>">
                                            <?php endif; ?>
                                            <div class="info-author-review">
                                                <?php if (!empty($item['reviewer_name'])) : ?>
                                                    <strong class="name-review"><?php echo genix_kses($item['reviewer_name']); ?></strong>
                                                <?php endif; ?>
                                                <?php if (!empty($item['reviewer_designation'])) : ?>
                                                    <p class="review-dept"><?php echo genix_kses($item['reviewer_designation']); ?></p>
                                                <?php endif; ?>
                                                <?php if (!empty($item['reviewer_brand']['url'])) : ?>
                                                    <div class="brand-review">
                                                        <img src="<?php echo esc_url($item['reviewer_brand']['url']) ?>" alt="<?php echo esc_attr__('Image', 'genixcore') ?>">
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="item-testimonial-right">
                                        <?php if (!empty($item['review_content'])) : ?>
                                            <p><?php echo esc_html($item['review_content']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <div class="swiper-pagination swiper-pagination-testimonials"></div>
                </div>
                <div class="testimonial__nav-four m-0">
                    <div class="testimonial-two-button-prev button-swiper-prev">
                        <i class="flaticon-right-arrow"></i>
                    </div>
                    <div class="testimonial-two-button-next button-swiper-next">
                        <i class="flaticon-right-arrow"></i>
                    </div>
                </div>
            </div>

        <?php elseif ($settings['tg_design_style']  == 'layout-6') : ?>

            <script>
                jQuery(document).ready(function($) {

                    var slider_test = new Swiper('.testiminials-active-2', {
                        slidesPerView: 2,
                        spaceBetween: 50,
                        loop: true,
                        navigation: {
                            nextEl: '.button-swiper-testimonial-next',
                            prevEl: '.button-swiper-testimonial-prev',
                        },
                        breakpoints: {
                            '1200': {
                                slidesPerView: 2,
                            },
                            '992': {
                                slidesPerView: 2,
                            },
                            '768': {
                                slidesPerView: 1,
                            },
                            '576': {
                                slidesPerView: 1,
                            },
                            '0': {
                                slidesPerView: 1,
                            },
                        },
                    });

                });
            </script>

            <div class="box-slide-testimonials">
                <div class="swiper-container testiminials-active-2">
                    <div class="swiper-wrapper">
                        <?php foreach ($settings['reviews_list'] as $item) :
                            if (!empty($item['reviewer_image']['url'])) {
                                $tg_reviewer_image = !empty($item['reviewer_image']['id']) ? wp_get_attachment_image_url($item['reviewer_image']['id'], $item['reviewer_image_size_size']) : $item['reviewer_image']['url'];
                                $tg_reviewer_image_alt = get_post_meta($item["reviewer_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                        ?>
                            <div class="swiper-slide">
                                <div class="card-testimonials">
                                    <?php if (!empty($item['reviewer_image'])) : ?>
                                        <div class="card-image">
                                            <img src="<?php echo esc_url($tg_reviewer_image); ?>" alt="<?php echo esc_url($tg_reviewer_image_alt); ?>">
                                        </div>
                                    <?php endif; ?>
                                    <div class="card-info">
                                        <?php if (!empty($item['reviewer_designation'])) : ?>
                                            <p class="card-position"><?php echo genix_kses($item['reviewer_designation']); ?></p>
                                        <?php endif; ?>
                                        <div class="testimonial__rating">
                                            <i class="fas fa-star"></i>
                                            <i class="fas fa-star"></i>
                                            <i class="fas fa-star"></i>
                                            <i class="fas fa-star"></i>
                                            <i class="fas fa-star"></i>
                                        </div>
                                        <?php if (!empty($item['review_content'])) : ?>
                                            <div class="card-comment">
                                                <p><?php echo esc_html($item['review_content']); ?></p>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>


        <?php else : ?>

            <div class="swiper-container testimonial-active">
                <div class="swiper-wrapper">
                    <?php foreach ($settings['reviews_list'] as $item) : ?>
                        <div class="swiper-slide">
                            <div class="testimonial-item">
                                <div class="testimonial-info">
                                    <?php if (!empty($item['reviewer_name'])) : ?>
                                        <h4 class="title"><?php echo genix_kses($item['reviewer_name']); ?></h4>
                                    <?php endif; ?>
                                    <?php if (!empty($item['reviewer_designation'])) : ?>
                                        <span><?php echo genix_kses($item['reviewer_designation']); ?></span>
                                    <?php endif; ?>
                                </div>
                                <div class="testimonial__rating">
                                    <i class="fas fa-star"></i>
                                    <i class="fas fa-star"></i>
                                    <i class="fas fa-star"></i>
                                    <i class="fas fa-star"></i>
                                    <i class="fas fa-star"></i>
                                </div>
                                <div class="testimonial-content">
                                    <?php if (!empty($item['review_content'])) : ?>
                                        <p><?php echo esc_html($item['review_content']); ?></p>
                                    <?php endif; ?>
                                    <div class="icon"><i class="fas fa-quote-right"></i></div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <div class="testimonial-slider-dot">
                <div class="swiper testimonial-nav">
                    <div class="swiper-wrapper">
                        <?php foreach ($settings['reviews_list'] as $item) :
                            if (!empty($item['reviewer_image']['url'])) {
                                $tg_reviewer_image = !empty($item['reviewer_image']['id']) ? wp_get_attachment_image_url($item['reviewer_image']['id'], $item['reviewer_image_size_size']) : $item['reviewer_image']['url'];
                                $tg_reviewer_image_alt = get_post_meta($item["reviewer_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                        ?>
                            <?php if (!empty($item['reviewer_image'])) : ?>
                                <div class="swiper-slide">
                                    <button><img src="<?php echo esc_url($tg_reviewer_image); ?>" alt="<?php echo esc_url($tg_reviewer_image_alt); ?>"></button>
                                </div>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>

        <?php endif; ?>

<?php
    }
}

$widgets_manager->register(new TG_Testimonial());
