<?php

namespace GenixCore\Widgets;

use Apexa_Navwalker_Class;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Control_Media;

use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Core\Schemes\Typography;
use \Elementor\Group_Control_Background;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Apexa Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class Genix_Header extends Widget_Base
{

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tg-header';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Header', 'genixcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'genix-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['genixcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['genixcore'];
    }

    protected $nav_menu_index = 1;

    /**
     * Retrieve the menu index.
     *
     * Used to get index of nav menu.
     *
     * @since 1.3.0
     * @access protected
     *
     * @return string nav index.
     */
    protected function get_nav_menu_index()
    {
        return $this->nav_menu_index++;
    }

    private function get_available_menus()
    {

        $menus = wp_get_nav_menus();

        $options = [];

        foreach ($menus as $menu) {
            $options[$menu->slug] = $menu->name;
        }

        return $options;
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {

        // layout Panel
        $this->start_controls_section(
            'genix_layout',
            [
                'label' => esc_html__('Design Layout', 'genixcore'),
            ]
        );

        $this->add_control(
            'tg_design_style',
            [
                'label' => esc_html__('Select Layout', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'genixcore'),
                    'layout-2' => esc_html__('Layout 2', 'genixcore'),
                    'layout-3' => esc_html__('Layout 3', 'genixcore'),
                    'layout-4' => esc_html__('Layout 4', 'genixcore'),
                    'layout-5' => esc_html__('Layout 5', 'genixcore'),
                    'layout-6' => esc_html__('Layout 6', 'genixcore'),
                    'layout-7' => esc_html__('Layout 7', 'genixcore'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        // Header Top
        $this->start_controls_section(
            '__genix_header_top',
            [
                'label' => esc_html__('Header Top', 'genixcore'),
                'condition' => [
                    'tg_design_style!' => ['layout-4', 'layout-6'],
                ],
            ]
        );

        $this->add_control(
            'header_top_show',
            [
                'label' => esc_html__('Show Header Top', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->start_controls_tabs(
            '_tab_style_top_box'
        );
        $this->start_controls_tab(
            '_tab_left_content',
            [
                'label' => esc_html__('Top Left', 'genixcore'),
                'condition' => [
                    'header_top_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'tg_phone_num',
            [
                'label' => esc_html__('Phone Number', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('+123 9898 500', 'genixcore'),
                'label_block' => true,
                'condition' => [
                    'header_top_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'tg_address',
            [
                'label' => esc_html__('Address', 'genixcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('256 Avenue, Mark Street, NewYork City', 'genixcore'),
                'label_block' => true,
                'condition' => [
                    'header_top_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            '_tab_right_content',
            [
                'label' => esc_html__('Top Right', 'genixcore'),
                'condition' => [
                    'header_top_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'tg_email_address',
            [
                'label' => esc_html__('Email Address', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('info@apexa.com', 'genixcore'),
                'label_block' => true,
                'condition' => [
                    'header_top_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'tg_open_time',
            [
                'label' => esc_html__('Open Time', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Mon-Fri: 10:00am - 09:00pm', 'genixcore'),
                'label_block' => true,
                'condition' => [
                    'header_top_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        // Logo
        $this->start_controls_section(
            '__genix_logo',
            [
                'label' => esc_html__('Logo', 'genixcore'),
            ]
        );

        $this->add_control(
            'genix_logo',
            [
                'label' => esc_html__('Choose Image', 'genixcore'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name'    => 'logo_size',
                'label'   => __('Image Size', 'genixcore'),
                'default' => 'thumb',
            ]
        );

        $this->end_controls_section();

        // Menu Style
        $this->start_controls_section(
            '__genix_menu-wrap',
            [
                'label' => esc_html__('Select Menu', 'genixcore'),
            ]
        );

        $menus = $this->get_available_menus();

        if (!empty($menus)) {
            $this->add_control(
                'menu',
                [
                    'label'        => __('Menu', 'genixcore'),
                    'type'         => Controls_Manager::SELECT,
                    'options'      => $menus,
                    'default'      => array_keys($menus)[0],
                    'save_default' => true,
                    /* translators: %s Nav menu URL */
                    'description'  => sprintf(__('Go to the <a href="%s" target="_blank">Menus screen</a> to manage your menus.', 'genixcore'), admin_url('nav-menus.php')),
                ]
            );
        } else {
            $this->add_control(
                'menu',
                [
                    'type' => Controls_Manager::RAW_HTML,
                    /* translators: %s Nav menu URL */
                    'raw' => sprintf(__('<strong>There are no menus in your site.</strong><br>Go to the <a href="%s" target="_blank">Menus screen</a> to create one.', 'genixcore'), admin_url('nav-menus.php?action=edit&menu=0')),
                    'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
                ]
            );
        }

        $this->add_control(
            'menu_last_item',
            [
                'label'     => __('Last Menu Item', 'genixcore'),
                'type'      => Controls_Manager::SELECT,
                'options'   => [
                    'none' => __('Default', 'genixcore'),
                    'cta'  => __('Button', 'genixcore'),
                ],
                'default'   => 'none',
                'condition' => [
                    'layout!' => 'expandible',
                ],
            ]
        );

        $this->end_controls_section();

        // Header Search
        $this->start_controls_section(
            '__genix_header_search',
            [
                'label' => esc_html__('Header Search', 'genixcore'),
                'condition' => [
                    'tg_design_style!' => ['layout-6', 'layout-7'],
                ]
            ]
        );

        $this->add_control(
            'header_search_show',
            [
                'label' => esc_html__('Show Header Search', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'header_search_text',
            [
                'label' => esc_html__('Search Placeholder Text', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Type keywords here', 'genixcore'),
                'label_block' => true,
            ]
        );

        $this->end_controls_section();

        // Header Offcanvas
        $this->start_controls_section(
            '__genix_header_offcanvas',
            [
                'label' => esc_html__('Header Offcanvas', 'genixcore'),
                'condition' => [
                    'tg_design_style!' => ['layout-6', 'layout-7'],
                ]
            ]
        );

        $this->add_control(
            'header_offcanvas_show',
            [
                'label' => esc_html__('Show Header Offcanvas', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->start_controls_tabs(
            '_tab_style_off_box'
        );
        $this->start_controls_tab(
            '_tab_info_content',
            [
                'label' => esc_html__('Contact Info', 'genixcore'),
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'off_title',
            [
                'label' => esc_html__('Title', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Office Address', 'genixcore'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'off_desc',
            [
                'label' => esc_html__('Description', 'genixcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('123/A, Miranda City Likaoli Prikano, Dope', 'genixcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'offcanvas_lists',
            [
                'label' => esc_html__('Item Lists', 'genixcore'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'off_title' => esc_html__('Office Address', 'genixcore'),
                    ],
                ],
                'title_field' => '{{{ off_title }}}',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            '_tab_social_content',
            [
                'label' => esc_html__('Social', 'genixcore'),
            ]
        );

        $this->add_control(
            'header_offcanvas_social',
            [
                'label' => esc_html__('Show Header Social', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $social = new \Elementor\Repeater();

        if (genix_is_elementor_version('<', '2.6.0')) {
            $social->add_control(
                'tg_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'fa fa-facebook',
                ]
            );
        } else {
            $social->add_control(
                'tg_selected_icon',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'label_block' => true,
                    'default' => [
                        'value' => 'fab fa-facebook-f',
                        'library' => 'solid',
                    ],
                ]
            );
        }

        $social->add_control(
            'tg_url',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => esc_html__('URL', 'genixcore'),
                'default' => esc_html__('#', 'genixcore'),
                'placeholder' => esc_html__('Type url here', 'genixcore'),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'social_lists',
            [
                'label' => esc_html__('Social Lists', 'genixcore'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $social->get_controls(),
                'default' => [
                    [
                        'tg_url' => esc_html__('#', 'genixcore'),
                    ],
                ],
                'condition' => [
                    'header_offcanvas_social' => 'yes'
                ]
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        // Header Button
        $this->start_controls_section(
            '__genix_header_button',
            [
                'label' => esc_html__('Header Button', 'genixcore'),
            ]
        );

        $this->add_control(
            'tg_button_show',
            [
                'label' => esc_html__('Show Button', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'tg_btn_text',
            [
                'label' => esc_html__('Button Text', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('let’s Talk', 'genixcore'),
                'title' => esc_html__('Enter button text', 'genixcore'),
                'label_block' => true,
                'condition' => [
                    'tg_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'tg_btn_link_type',
            [
                'label' => esc_html__('Button Link Type', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
                'condition' => [
                    'tg_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'tg_btn_link',
            [
                'label' => esc_html__('Button link', 'genixcore'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'genixcore'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'tg_btn_link_type' => '1',
                    'tg_button_show' => 'yes'
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'tg_btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'genixcore'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => genix_get_all_pages(),
                'condition' => [
                    'tg_btn_link_type' => '2',
                    'tg_button_show' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'tg_button_show02',
            [
                'label' => esc_html__('Show Button 02', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
                'condition' => [
                    'tg_design_style' => ['layout-6', 'layout-7'],
                ]
            ]
        );

        $this->add_control(
            'tg_btn_text02',
            [
                'label' => esc_html__('Button Text', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Free Trial', 'genixcore'),
                'title' => esc_html__('Enter button text', 'genixcore'),
                'label_block' => true,
                'condition' => [
                    'tg_button_show02' => 'yes',
                    'tg_design_style' => ['layout-6', 'layout-7'],
                ],
            ]
        );

        $this->add_control(
            'tg_btn_link_type02',
            [
                'label' => esc_html__('Button Link Type', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
                'condition' => [
                    'tg_button_show02' => 'yes',
                    'tg_design_style' => ['layout-6', 'layout-7'],
                ],
            ]
        );

        $this->add_control(
            'tg_btn_link02',
            [
                'label' => esc_html__('Button link', 'genixcore'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'genixcore'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'tg_btn_link_type02' => '1',
                    'tg_button_show02' => 'yes',
                    'tg_design_style' => ['layout-6', 'layout-7'],
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'tg_btn_page_link02',
            [
                'label' => esc_html__('Select Button Page', 'genixcore'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => genix_get_all_pages(),
                'condition' => [
                    'tg_btn_link_type02' => '2',
                    'tg_button_show02' => 'yes',
                    'tg_design_style' => ['layout-6', 'layout-7'],
                ]
            ]
        );

        $this->end_controls_section();

        // Mobile Menu
        $this->start_controls_section(
            '__genix_header_mobile',
            [
                'label' => esc_html__('Mobile Menu', 'genixcore'),
            ]
        );

        $this->add_control(
            'mobile_search_show',
            [
                'label' => esc_html__('Show Mobile Search', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'mobile_search_text',
            [
                'label' => esc_html__('Search Placeholder Text', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Search here...', 'genixcore'),
                'label_block' => true,
                'condition' => [
                    'mobile_search_show' => 'yes'
                ]
            ]
        );

        $this->start_controls_tabs(
            '_tab_style_mobile_box'
        );
        $this->start_controls_tab(
            '_tab_mobile_info_content',
            [
                'label' => esc_html__('Contact Info', 'genixcore'),
            ]
        );

        $this->add_control(
            'mobile_email',
            [
                'label' => esc_html__('Email Address', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('info@apexa.com', 'genixcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'mobile_phone',
            [
                'label' => esc_html__('Phone Number', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('+123 888 9999', 'genixcore'),
                'label_block' => true,
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            '_tab_mobile_social',
            [
                'label' => esc_html__('Social List', 'genixcore'),
            ]
        );

        $this->add_control(
            'mobile_social_show',
            [
                'label' => esc_html__('Show Mobile Social', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $social2 = new \Elementor\Repeater();

        if (genix_is_elementor_version('<', '2.6.0')) {
            $social2->add_control(
                'tg_icon2',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICON,
                    'label_block' => true,
                    'default' => 'fa fa-facebook',
                ]
            );
        } else {
            $social2->add_control(
                'tg_selected_icon2',
                [
                    'show_label' => false,
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'label_block' => true,
                    'default' => [
                        'value' => 'fab fa-facebook-f',
                        'library' => 'solid',
                    ],
                ]
            );
        }

        $social2->add_control(
            'tg_url2',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => esc_html__('URL', 'genixcore'),
                'default' => esc_html__('#', 'genixcore'),
                'placeholder' => esc_html__('Type url here', 'genixcore'),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'mobile_socials',
            [
                'label' => esc_html__('Social Lists', 'genixcore'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $social2->get_controls(),
                'default' => [
                    [
                        'tg_url2' => esc_html__('#', 'genixcore'),
                    ],
                ],
                'condition' => [
                    'mobile_social_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        // TAB_STYLE
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'genixcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'sticky_menu_show',
            [
                'label' => esc_html__('Show Sticky Header', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'sticky-header-none',
            ]
        );

        $this->end_controls_section();

        // Style Tab Here
        $this->start_controls_section(
            '_section_style_content',
            [
                'label' => esc_html__('Menu Style', 'genixcore'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs(
            '_tab_style_menu_box'
        );
        $this->start_controls_tab(
            '_tab_menu_normal',
            [
                'label' => esc_html__('Normal', 'genixcore'),
            ]
        );

        $this->add_control(
            'menu_color',
            [
                'label' => esc_html__('Text Color', 'genixcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tgmenu__navbar-wrap > ul > li a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            '_tab_menu_sticky',
            [
                'label' => esc_html__('Sticky', 'genixcore'),
            ]
        );

        $this->add_control(
            'sticky_menu_color',
            [
                'label' => esc_html__('Text Color', 'genixcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .sticky-menu .tgmenu__navbar-wrap > ul > li a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            '_tab_menu_hover',
            [
                'label' => esc_html__('Hover', 'genixcore'),
            ]
        );

        $this->add_control(
            'hover_menu_color',
            [
                'label' => esc_html__('Text Color', 'genixcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tgmenu__navbar-wrap > ul > li:hover a' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .tgmenu__navbar-wrap > ul > li.active a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->add_control(
            '_menu_gap',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__('Menu Gap', 'genixcore'),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'item_spacing',
            [
                'label' => esc_html__('Item Spacing', 'genixcore'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .tgmenu__navbar-wrap > ul > li' => 'margin-right: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $menus = $this->get_available_menus();


        if (isset($settings['menu'])) {
            require_once get_parent_theme_file_path() . '/inc/class-navwalker.php';
            $args = [
                'echo'        => false,
                'menu'        => $settings['menu'],
                'menu_class'  => 'navigation',
                'menu_id'     => 'menu-' . $this->get_nav_menu_index() . '-' . $this->get_id(),
                'fallback_cb' => 'Apexa_Navwalker_Class::fallback',
                'container'   => '',
                'walker'      => new Apexa_Navwalker_Class,
            ];
            $menu_html = wp_nav_menu($args);
        }

        // Logo Size
        $size = $settings['logo_size_size'];

        if ('custom' !== $size) {
            $image_size = $size;
        } else {
            require_once ELEMENTOR_PATH . 'includes/libraries/bfi-thumb/bfi-thumb.php';
            $image_dimension = $settings['logo_size_custom_dimension'];
            $image_size = [
                0           => null, // Width.
                1           => null, // Height.
                'bfi_thumb' => true,
                'crop'      => true,
            ];
            $has_custom_size = false;
            if (!empty($image_dimension['width'])) {
                $has_custom_size = true;
                $image_size[0]   = $image_dimension['width'];
            }
            if (!empty($image_dimension['height'])) {
                $has_custom_size = true;
                $image_size[1]   = $image_dimension['height'];
            }
            if (!$has_custom_size) {
                $image_size = 'full';
            }
        }

        if (!empty($settings['genix_logo']['url'])) {
            $genix_logo = !empty($settings['genix_logo']['id']) ? wp_get_attachment_image_url($settings['genix_logo']['id'], $image_size, true) : $settings['genix_logo']['url'];
            $genix_alt = get_post_meta($settings["genix_logo"]["id"], "_wp_attachment_image_alt", true);
        }

        $sticky_header = $settings['sticky_menu_show'] == 'yes' ? 'sticky-header' : 'sticky-header-none';

        // Link
        if ('2' == $settings['tg_btn_link_type']) {
            $this->add_render_attribute('tg-button-arg', 'href', get_permalink($settings['tg_btn_page_link']));
            $this->add_render_attribute('tg-button-arg', 'target', '_self');
            $this->add_render_attribute('tg-button-arg', 'rel', 'nofollow');
            $this->add_render_attribute('tg-button-arg', 'class', 'btn');
        } else {
            if (!empty($settings['tg_btn_link']['url'])) {
                $this->add_link_attributes('tg-button-arg', $settings['tg_btn_link']);
                $this->add_render_attribute('tg-button-arg', 'class', 'btn');
            }
        } ?>


        <?php if ($settings['tg_design_style'] == 'layout-2') : ?>

            <div class="transparent-header">
                <?php if (!empty($settings['header_top_show'])) : ?>
                    <div class="tg-header__top">
                        <div class="container custom-container">
                            <div class="row justify-content-center">
                                <div class="col-lg-6">
                                    <ul class="tg-header__top-info left-side list-wrap">
                                        <?php if (!empty($settings['tg_phone_num'])) : ?>
                                            <li>
                                                <i class="flaticon-phone-call"></i>
                                                <a href="tel:<?php echo esc_attr($settings['tg_phone_num']); ?>"><?php echo genix_kses($settings['tg_phone_num']); ?></a>
                                            </li>
                                        <?php endif; ?>
                                        <?php if (!empty($settings['tg_address'])) : ?>
                                            <li>
                                                <i class="flaticon-pin"></i>
                                                <?php echo genix_kses($settings['tg_address']); ?>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </div>
                                <div class="col-lg-6">
                                    <ul class="tg-header__top-right list-wrap">
                                        <?php if (!empty($settings['tg_email_address'])) : ?>
                                            <li>
                                                <i class="flaticon-envelope"></i>
                                                <a href="mailto:<?php echo esc_attr($settings['tg_email_address']); ?>"><?php echo genix_kses($settings['tg_email_address']); ?></a>
                                            </li>
                                        <?php endif; ?>
                                        <?php if (!empty($settings['tg_open_time'])) : ?>
                                            <li>
                                                <i class="flaticon-time"></i>
                                                <?php echo genix_kses($settings['tg_open_time']); ?>
                                            </li>
                                        <?php endif; ?>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                <div id="<?php echo esc_attr($sticky_header); ?>" class="tg-header__area">
                    <div class="container custom-container">
                        <div class="row">
                            <div class="col-12">
                                <div class="tgmenu__wrap">
                                    <nav class="tgmenu__nav">
                                        <?php if (!empty($genix_logo)) : ?>
                                            <div class="logo">
                                                <a href="<?php print esc_url(home_url('/')); ?>">
                                                    <img src="<?php echo esc_url($genix_logo); ?>" alt="<?php echo esc_attr($genix_alt); ?>">
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                        <div class="tgmenu__navbar-wrap tgmenu__main-menu d-none d-lg-flex">
                                            <?php echo isset($menu_html) ? $menu_html : ''; ?>
                                        </div>

                                        <?php if (!empty($settings['header_search_show'] || $settings['header_offcanvas_show'] || $settings['tg_button_show'])) : ?>
                                            <div class="tgmenu__action d-none d-md-block">
                                                <ul class="list-wrap">
                                                    <?php if (!empty($settings['header_search_show'])) : ?>
                                                        <li class="header-search">
                                                            <a href="javascript:void(0)" class="search-open-btn">
                                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="none">
                                                                    <path d="M19 19.0002L14.657 14.6572M14.657 14.6572C15.3999 13.9143 15.9892 13.0324 16.3912 12.0618C16.7933 11.0911 17.0002 10.0508 17.0002 9.00021C17.0002 7.9496 16.7933 6.90929 16.3913 5.93866C15.9892 4.96803 15.3999 4.08609 14.657 3.34321C13.9141 2.60032 13.0322 2.01103 12.0616 1.60898C11.0909 1.20693 10.0506 1 9.00002 1C7.94942 1 6.90911 1.20693 5.93848 1.60898C4.96785 2.01103 4.08591 2.60032 3.34302 3.34321C1.84269 4.84354 0.999817 6.87842 0.999817 9.00021C0.999817 11.122 1.84269 13.1569 3.34302 14.6572C4.84335 16.1575 6.87824 17.0004 9.00002 17.0004C11.1218 17.0004 13.1567 16.1575 14.657 14.6572Z" stroke="currentcolor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" />
                                                                </svg>
                                                            </a>
                                                        </li>
                                                    <?php endif; ?>

                                                    <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                                                        <li class="offCanvas-menu">
                                                            <a href="javascript:void(0)" class="menu-tigger">
                                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                                    <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                                    <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                                    <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                                    <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                                    <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                                    <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                                    <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                                    <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                                    <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                                </svg>
                                                            </a>
                                                        </li>
                                                    <?php endif; ?>

                                                    <?php if (!empty($settings['tg_button_show'])) : ?>
                                                        <li class="header-btn">
                                                            <a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>>
                                                                <?php echo $settings['tg_btn_text']; ?>
                                                            </a>
                                                        </li>
                                                    <?php endif; ?>
                                                </ul>
                                            </div>
                                        <?php endif; ?>

                                        <?php if (has_nav_menu('main-menu')) { ?>
                                            <div class="mobile-nav-toggler">
                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                    <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                    <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                    <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                    <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                    <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                    <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                    <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                    <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                    <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                </svg>
                                            </div>
                                        <?php } ?>
                                    </nav>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/mobile-menu.php'); ?>

                <?php if (!empty($settings['header_search_show'])) : ?>
                    <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/header-search.php'); ?>
                <?php endif; ?>

                <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                    <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/offcanvas.php'); ?>
                <?php endif; ?>

            </div>

        <?php elseif ($settings['tg_design_style'] == 'layout-3') : ?>

            <div id="header-fixed-height"></div>
            <div class="tg-header__style-two">
                <div class="tg-header__inner-wrap">
                    <?php if (!empty($genix_logo)) : ?>
                        <div class="tg-header__logo-wrap">
                            <div class="logo">
                                <a href="<?php print esc_url(home_url('/')); ?>">
                                    <img src="<?php echo esc_url($genix_logo); ?>" alt="<?php echo esc_attr($genix_alt); ?>">
                                </a>
                            </div>
                        </div>
                    <?php endif; ?>
                    <div class="tg-header__right-side">
                        <?php if (!empty($settings['header_top_show'])) : ?>
                            <div class="tg-header__top tg-header__top-two">
                                <div class="row">
                                    <div class="col-lg-6">
                                        <ul class="tg-header__top-info tg-header__top-info-two list-wrap">
                                            <?php if (!empty($settings['tg_phone_num'])) : ?>
                                                <li><i class="flaticon-phone-call"></i>
                                                    <a href="tel:<?php echo esc_attr($settings['tg_phone_num']); ?>"><?php echo genix_kses($settings['tg_phone_num']); ?></a>
                                                </li>
                                            <?php endif; ?>
                                            <?php if (!empty($settings['tg_address'])) : ?>
                                                <li><i class="flaticon-pin"></i><?php echo genix_kses($settings['tg_address']); ?></li>
                                            <?php endif; ?>
                                        </ul>
                                    </div>
                                    <div class="col-lg-6">
                                        <ul class="tg-header__top-right tg-header__top-right-two list-wrap">
                                            <?php if (!empty($settings['tg_email_address'])) : ?>
                                                <li><i class="flaticon-envelope"></i><a href="mailto:<?php echo esc_attr($settings['tg_email_address']); ?>"><?php echo genix_kses($settings['tg_email_address']); ?></a></li>
                                            <?php endif; ?>
                                            <?php if (!empty($settings['tg_open_time'])) : ?>
                                                <li><i class="flaticon-time"></i><?php echo genix_kses($settings['tg_open_time']); ?></li>
                                            <?php endif; ?>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                        <div id="<?php echo esc_attr($sticky_header); ?>" class="tg-header__area tg-header__area-two">
                            <div class="row">
                                <div class="col-12">
                                    <div class="tgmenu__wrap">
                                        <nav class="tgmenu__nav">
                                            <?php if (!empty($genix_logo)) : ?>
                                                <div class="logo d-none">
                                                    <a href="<?php print esc_url(home_url('/')); ?>">
                                                        <img src="<?php echo esc_url($genix_logo); ?>" alt="<?php echo esc_attr($genix_alt); ?>">
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                            <div class="tgmenu__navbar-wrap tgmenu__main-menu d-none d-lg-flex">
                                                <?php echo isset($menu_html) ? $menu_html : ''; ?>
                                            </div>
                                            <?php if (!empty($settings['header_search_show'] || $settings['header_offcanvas_show'] || $settings['tg_button_show'])) : ?>
                                                <div class="tgmenu__action tgmenu__action-two d-none d-md-block">
                                                    <ul class="list-wrap">
                                                        <?php if (!empty($settings['header_search_show'])) : ?>
                                                            <li class="header-search">
                                                                <a href="javascript:void(0)" class="search-open-btn">
                                                                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="none">
                                                                        <path d="M19 19.0002L14.657 14.6572M14.657 14.6572C15.3999 13.9143 15.9892 13.0324 16.3912 12.0618C16.7933 11.0911 17.0002 10.0508 17.0002 9.00021C17.0002 7.9496 16.7933 6.90929 16.3913 5.93866C15.9892 4.96803 15.3999 4.08609 14.657 3.34321C13.9141 2.60032 13.0322 2.01103 12.0616 1.60898C11.0909 1.20693 10.0506 1 9.00002 1C7.94942 1 6.90911 1.20693 5.93848 1.60898C4.96785 2.01103 4.08591 2.60032 3.34302 3.34321C1.84269 4.84354 0.999817 6.87842 0.999817 9.00021C0.999817 11.122 1.84269 13.1569 3.34302 14.6572C4.84335 16.1575 6.87824 17.0004 9.00002 17.0004C11.1218 17.0004 13.1567 16.1575 14.657 14.6572Z" stroke="currentcolor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" />
                                                                    </svg>
                                                                </a>
                                                            </li>
                                                        <?php endif; ?>
                                                        <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                                                            <li class="offCanvas-menu">
                                                                <a href="javascript:void(0)" class="menu-tigger">
                                                                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                                        <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                                        <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                                        <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                                        <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                                        <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                                        <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                                        <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                                        <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                                        <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                                    </svg>
                                                                </a>
                                                            </li>
                                                        <?php endif; ?>
                                                        <?php if (!empty($settings['tg_button_show'])) : ?>
                                                            <li class="header-btn"><a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>><i class="far fa-comment-alt"></i><?php echo $settings['tg_btn_text']; ?></a></li>
                                                        <?php endif; ?>
                                                    </ul>
                                                </div>
                                            <?php endif; ?>
                                            <?php if (has_nav_menu('main-menu')) { ?>
                                                <div class="mobile-nav-toggler mobile-nav-toggler-two">
                                                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                        <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                        <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                        <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                        <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                        <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                        <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                        <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                        <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                        <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                    </svg>
                                                </div>
                                            <?php } ?>
                                        </nav>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/mobile-menu.php'); ?>

                <?php if (!empty($settings['header_search_show'])) : ?>
                    <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/header-search.php'); ?>
                <?php endif; ?>

                <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                    <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/offcanvas.php'); ?>
                <?php endif; ?>

            </div>

        <?php elseif ($settings['tg_design_style'] == 'layout-4') : ?>

            <div class="tg-header__style-three transparent-header">
                <div id="<?php echo esc_attr($sticky_header); ?>" class="tg-header__area tg-header__area-three">
                    <div class="container custom-container">
                        <div class="row">
                            <div class="col-12">
                                <div class="tgmenu__wrap">
                                    <nav class="tgmenu__nav">
                                        <?php if (!empty($genix_logo)) : ?>
                                            <div class="logo">
                                                <a href="<?php print esc_url(home_url('/')); ?>">
                                                    <img src="<?php echo esc_url($genix_logo); ?>" alt="<?php echo esc_attr($genix_alt); ?>">
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                        <div class="tgmenu__navbar-wrap tgmenu__main-menu d-none d-lg-flex">
                                            <?php echo isset($menu_html) ? $menu_html : ''; ?>
                                        </div>
                                        <?php if (!empty($settings['header_search_show'] || $settings['header_offcanvas_show'] || $settings['tg_button_show'])) : ?>
                                            <div class="tgmenu__action tgmenu__action-three d-none d-md-block">
                                                <ul class="list-wrap">
                                                    <?php if (!empty($settings['header_search_show'])) : ?>
                                                        <li class="header-search">
                                                            <a href="javascript:void(0)" class="search-open-btn">
                                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="none">
                                                                    <path d="M19 19.0002L14.657 14.6572M14.657 14.6572C15.3999 13.9143 15.9892 13.0324 16.3912 12.0618C16.7933 11.0911 17.0002 10.0508 17.0002 9.00021C17.0002 7.9496 16.7933 6.90929 16.3913 5.93866C15.9892 4.96803 15.3999 4.08609 14.657 3.34321C13.9141 2.60032 13.0322 2.01103 12.0616 1.60898C11.0909 1.20693 10.0506 1 9.00002 1C7.94942 1 6.90911 1.20693 5.93848 1.60898C4.96785 2.01103 4.08591 2.60032 3.34302 3.34321C1.84269 4.84354 0.999817 6.87842 0.999817 9.00021C0.999817 11.122 1.84269 13.1569 3.34302 14.6572C4.84335 16.1575 6.87824 17.0004 9.00002 17.0004C11.1218 17.0004 13.1567 16.1575 14.657 14.6572Z" stroke="currentcolor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" />
                                                                </svg>
                                                            </a>
                                                        </li>
                                                    <?php endif; ?>
                                                    <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                                                        <li class="offCanvas-menu">
                                                            <a href="javascript:void(0)" class="menu-tigger">
                                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                                    <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                                    <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                                    <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                                    <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                                    <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                                    <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                                    <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                                    <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                                    <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                                </svg>
                                                            </a>
                                                        </li>
                                                    <?php endif; ?>
                                                    <?php if (!empty($settings['tg_button_show'])) : ?>
                                                        <li class="header-btn"><a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>><?php echo $settings['tg_btn_text']; ?></a></li>
                                                    <?php endif; ?>
                                                </ul>
                                            </div>
                                        <?php endif; ?>
                                        <?php if (has_nav_menu('main-menu')) { ?>
                                            <div class="mobile-nav-toggler mobile-nav-toggler-two">
                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                    <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                    <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                    <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                    <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                    <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                    <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                    <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                    <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                    <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                </svg>
                                            </div>
                                        <?php } ?>
                                    </nav>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/mobile-menu.php'); ?>

                <?php if (!empty($settings['header_search_show'])) : ?>
                    <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/header-search.php'); ?>
                <?php endif; ?>

                <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                    <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/offcanvas.php'); ?>
                <?php endif; ?>

            </div>

        <?php elseif ($settings['tg_design_style'] == 'layout-5') : ?>

            <div class="tg-header__style-four transparent-header">
                <?php if (!empty($settings['header_top_show'])) : ?>
                    <div class="tg-header__top tg-header__top-three">
                        <div class="container">
                            <div class="row justify-content-center">
                                <div class="col-lg-6">
                                    <ul class="tg-header__top-info tg-header__top-info-three list-wrap">
                                        <?php if (!empty($settings['tg_phone_num'])) : ?>
                                            <li><i class="flaticon-phone-call"></i><a href="tel:<?php echo esc_attr($settings['tg_phone_num']); ?>"><?php echo genix_kses($settings['tg_phone_num']); ?></a></li>
                                        <?php endif; ?>
                                        <?php if (!empty($settings['tg_address'])) : ?>
                                            <li><i class="flaticon-pin"></i><?php echo genix_kses($settings['tg_address']); ?></li>
                                        <?php endif; ?>
                                    </ul>
                                </div>
                                <div class="col-lg-6">
                                    <ul class="tg-header__top-right tg-header__top-right-three list-wrap">
                                        <?php if (!empty($settings['tg_email_address'])) : ?>
                                            <li><i class="flaticon-envelope"></i><a href="mailto:<?php echo esc_attr($settings['tg_email_address']); ?>"><?php echo genix_kses($settings['tg_email_address']); ?></a></li>
                                        <?php endif; ?>
                                        <?php if (!empty($settings['tg_open_time'])) : ?>
                                            <li><i class="flaticon-time"></i><?php echo genix_kses($settings['tg_open_time']); ?></li>
                                        <?php endif; ?>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
                <div id="<?php echo esc_attr($sticky_header); ?>" class="tg-header__area tg-header__area-four">
                    <div class="container">
                        <div class="row">
                            <div class="col-12">
                                <div class="tgmenu__wrap">
                                    <nav class="tgmenu__nav">
                                        <?php if (!empty($genix_logo)) : ?>
                                            <div class="logo">
                                                <a href="<?php print esc_url(home_url('/')); ?>">
                                                    <img src="<?php echo esc_url($genix_logo); ?>" alt="<?php echo esc_attr($genix_alt); ?>">
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                        <div class="tgmenu__navbar-wrap tgmenu__main-menu d-none d-lg-flex">
                                            <?php echo isset($menu_html) ? $menu_html : ''; ?>
                                        </div>
                                        <?php if (!empty($settings['header_search_show'] || $settings['header_offcanvas_show'] || $settings['tg_button_show'])) : ?>
                                            <div class="tgmenu__action tgmenu__action-four d-none d-md-block">
                                                <ul class="list-wrap">
                                                    <?php if (!empty($settings['header_search_show'])) : ?>
                                                        <li class="header-search">
                                                            <a href="javascript:void(0)" class="search-open-btn">
                                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="none">
                                                                    <path d="M19 19.0002L14.657 14.6572M14.657 14.6572C15.3999 13.9143 15.9892 13.0324 16.3912 12.0618C16.7933 11.0911 17.0002 10.0508 17.0002 9.00021C17.0002 7.9496 16.7933 6.90929 16.3913 5.93866C15.9892 4.96803 15.3999 4.08609 14.657 3.34321C13.9141 2.60032 13.0322 2.01103 12.0616 1.60898C11.0909 1.20693 10.0506 1 9.00002 1C7.94942 1 6.90911 1.20693 5.93848 1.60898C4.96785 2.01103 4.08591 2.60032 3.34302 3.34321C1.84269 4.84354 0.999817 6.87842 0.999817 9.00021C0.999817 11.122 1.84269 13.1569 3.34302 14.6572C4.84335 16.1575 6.87824 17.0004 9.00002 17.0004C11.1218 17.0004 13.1567 16.1575 14.657 14.6572Z" stroke="currentcolor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" />
                                                                </svg>
                                                            </a>
                                                        </li>
                                                    <?php endif; ?>
                                                    <?php if (!empty($settings['tg_button_show'])) : ?>
                                                        <li class="header-btn"><a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>><?php echo $settings['tg_btn_text']; ?></a></li>
                                                    <?php endif; ?>
                                                    <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                                                        <li class="offCanvas-menu">
                                                            <a href="javascript:void(0)" class="menu-tigger">
                                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                                    <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                                    <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                                    <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                                    <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                                    <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                                    <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                                    <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                                    <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                                    <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                                </svg>
                                                            </a>
                                                        </li>
                                                    <?php endif; ?>
                                                </ul>
                                            </div>
                                        <?php endif; ?>
                                        <?php if (has_nav_menu('main-menu')) { ?>
                                            <div class="mobile-nav-toggler mobile-nav-toggler-two">
                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                    <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                    <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                    <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                    <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                    <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                    <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                    <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                    <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                    <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                </svg>
                                            </div>
                                        <?php } ?>
                                    </nav>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/mobile-menu.php'); ?>

                <?php if (!empty($settings['header_search_show'])) : ?>
                    <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/header-search.php'); ?>
                <?php endif; ?>

                <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                    <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/offcanvas.php'); ?>
                <?php endif; ?>

            </div>

        <?php elseif ($settings['tg_design_style'] == 'layout-6') :

            // Link
            if ('2' == $settings['tg_btn_link_type02']) {
                $this->add_render_attribute('tg-button-arg02', 'href', get_permalink($settings['tg_btn_page_link02']));
                $this->add_render_attribute('tg-button-arg02', 'target', '_self');
                $this->add_render_attribute('tg-button-arg02', 'rel', 'nofollow');
                $this->add_render_attribute('tg-button-arg02', 'class', 'btn border-btn');
            } else {
                if (!empty($settings['tg_btn_link02']['url'])) {
                    $this->add_link_attributes('tg-button-arg02', $settings['tg_btn_link02']);
                    $this->add_render_attribute('tg-button-arg02', 'class', 'btn border-btn');
                }
            }
        ?>

            <div class="tg-header__style-five transparent-header">
                <div id="<?php echo esc_attr($sticky_header); ?>" class="tg-header__area tg-header__area-five">
                    <div class="container">
                        <div class="row">
                            <div class="col-12">
                                <div class="tgmenu__wrap">
                                    <nav class="tgmenu__nav">
                                        <?php if (!empty($genix_logo)) : ?>
                                            <div class="logo">
                                                <a href="<?php print esc_url(home_url('/')); ?>">
                                                    <img src="<?php echo esc_url($genix_logo); ?>" alt="<?php echo esc_attr($genix_alt); ?>">
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                        <div class="tgmenu__navbar-wrap tgmenu__main-menu d-none d-lg-flex">
                                            <?php echo isset($menu_html) ? $menu_html : ''; ?>
                                        </div>
                                        <?php if (!empty($settings['tg_button_show'] || $settings['tg_button_show02'])) : ?>
                                            <div class="tgmenu__action tgmenu__action-five d-none d-md-block">
                                                <ul class="list-wrap">
                                                    <?php if (!empty($settings['tg_button_show02'])) : ?>
                                                        <li class="header-btn-two">
                                                            <a <?php echo $this->get_render_attribute_string('tg-button-arg02'); ?>><?php echo $settings['tg_btn_text02']; ?></a>
                                                        </li>
                                                    <?php endif; ?>
                                                    <?php if (!empty($settings['tg_button_show'])) : ?>
                                                        <li class="header-btn">
                                                            <a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>><?php echo $settings['tg_btn_text']; ?></a>
                                                        </li>
                                                    <?php endif; ?>
                                                </ul>
                                            </div>
                                        <?php endif; ?>
                                        <?php if (has_nav_menu('main-menu')) { ?>
                                            <div class="mobile-nav-toggler mobile-nav-toggler-two">
                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                    <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                    <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                    <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                    <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                    <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                    <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                    <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                    <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                    <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                </svg>
                                            </div>
                                        <?php } ?>
                                    </nav>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/mobile-menu.php'); ?>

            </div>

        <?php elseif ($settings['tg_design_style'] == 'layout-7') :

            // Link
            if ('2' == $settings['tg_btn_link_type02']) {
                $this->add_render_attribute('tg-button-arg02', 'href', get_permalink($settings['tg_btn_page_link02']));
                $this->add_render_attribute('tg-button-arg02', 'target', '_self');
                $this->add_render_attribute('tg-button-arg02', 'rel', 'nofollow');
                $this->add_render_attribute('tg-button-arg02', 'class', 'btn border-btn');
            } else {
                if (!empty($settings['tg_btn_link02']['url'])) {
                    $this->add_link_attributes('tg-button-arg02', $settings['tg_btn_link02']);
                    $this->add_render_attribute('tg-button-arg02', 'class', 'btn border-btn');
                }
            }
        ?>

            <?php if (!empty($settings['header_top_show'])) : ?>
                <div class="tg-header__top">
                    <div class="container">
                        <div class="row justify-content-center">
                            <div class="col-lg-6">
                                <ul class="tg-header__top-info left-side list-wrap">
                                    <?php if (!empty($settings['tg_phone_num'])) : ?>
                                        <li>
                                            <i class="flaticon-phone-call"></i>
                                            <a href="tel:<?php echo esc_attr($settings['tg_phone_num']); ?>"><?php echo genix_kses($settings['tg_phone_num']); ?></a>
                                        </li>
                                    <?php endif; ?>
                                    <?php if (!empty($settings['tg_address'])) : ?>
                                        <li>
                                            <i class="flaticon-pin"></i>
                                            <?php echo genix_kses($settings['tg_address']); ?>
                                        </li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                            <div class="col-lg-6">
                                <ul class="tg-header__top-right list-wrap">
                                    <?php if (!empty($settings['tg_email_address'])) : ?>
                                        <li>
                                            <i class="flaticon-envelope"></i>
                                            <a href="mailto:<?php echo esc_attr($settings['tg_email_address']); ?>"><?php echo genix_kses($settings['tg_email_address']); ?></a>
                                        </li>
                                    <?php endif; ?>
                                    <?php if (!empty($settings['tg_open_time'])) : ?>
                                        <li>
                                            <i class="flaticon-time"></i>
                                            <?php echo genix_kses($settings['tg_open_time']); ?>
                                        </li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
            <div class="tg-header__style-five">
                <div id="header-fixed-height"></div>
                <div id="<?php echo esc_attr($sticky_header); ?>" class="tg-header__area tg-header__area-five">
                    <div class="container">
                        <div class="row">
                            <div class="col-12">
                                <div class="tgmenu__wrap">
                                    <nav class="tgmenu__nav">
                                        <?php if (!empty($genix_logo)) : ?>
                                            <div class="logo">
                                                <a href="<?php print esc_url(home_url('/')); ?>">
                                                    <img src="<?php echo esc_url($genix_logo); ?>" alt="<?php echo esc_attr($genix_alt); ?>">
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                        <div class="tgmenu__navbar-wrap tgmenu__main-menu d-none d-lg-flex">
                                            <?php echo isset($menu_html) ? $menu_html : ''; ?>
                                        </div>
                                        <?php if (!empty($settings['tg_button_show'] || $settings['tg_button_show02'])) : ?>
                                            <div class="tgmenu__action tgmenu__action-five d-none d-md-block">
                                                <ul class="list-wrap">
                                                    <?php if (!empty($settings['tg_button_show02'])) : ?>
                                                        <li class="header-btn-two">
                                                            <a <?php echo $this->get_render_attribute_string('tg-button-arg02'); ?>><?php echo $settings['tg_btn_text02']; ?></a>
                                                        </li>
                                                    <?php endif; ?>
                                                    <?php if (!empty($settings['tg_button_show'])) : ?>
                                                        <li class="header-btn">
                                                            <a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>><?php echo $settings['tg_btn_text']; ?></a>
                                                        </li>
                                                    <?php endif; ?>
                                                </ul>
                                            </div>
                                        <?php endif; ?>
                                        <?php if (has_nav_menu('main-menu')) { ?>
                                            <div class="mobile-nav-toggler mobile-nav-toggler-two">
                                                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                    <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                    <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                    <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                    <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                    <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                    <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                    <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                    <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                    <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                </svg>
                                            </div>
                                        <?php } ?>
                                    </nav>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/mobile-menu.php'); ?>
            </div>


        <?php else : ?>

            <?php if (!empty($settings['header_top_show'])) : ?>
                <div class="tg-header__top">
                    <div class="container custom-container">
                        <div class="row justify-content-center">
                            <div class="col-lg-6">
                                <ul class="tg-header__top-info left-side list-wrap">
                                    <?php if (!empty($settings['tg_phone_num'])) : ?>
                                        <li>
                                            <i class="flaticon-phone-call"></i>
                                            <a href="tel:<?php echo esc_attr($settings['tg_phone_num']); ?>"><?php echo genix_kses($settings['tg_phone_num']); ?></a>
                                        </li>
                                    <?php endif; ?>
                                    <?php if (!empty($settings['tg_address'])) : ?>
                                        <li>
                                            <i class="flaticon-pin"></i>
                                            <?php echo genix_kses($settings['tg_address']); ?>
                                        </li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                            <div class="col-lg-6">
                                <ul class="tg-header__top-right list-wrap">
                                    <?php if (!empty($settings['tg_email_address'])) : ?>
                                        <li>
                                            <i class="flaticon-envelope"></i>
                                            <a href="mailto:<?php echo esc_attr($settings['tg_email_address']); ?>"><?php echo genix_kses($settings['tg_email_address']); ?></a>
                                        </li>
                                    <?php endif; ?>
                                    <?php if (!empty($settings['tg_open_time'])) : ?>
                                        <li>
                                            <i class="flaticon-time"></i>
                                            <?php echo genix_kses($settings['tg_open_time']); ?>
                                        </li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
            <div id="<?php echo esc_attr($sticky_header); ?>" class="tg-header__area">
                <div class="container custom-container">
                    <div class="row">
                        <div class="col-12">
                            <div class="tgmenu__wrap">
                                <nav class="tgmenu__nav">
                                    <?php if (!empty($genix_logo)) : ?>
                                        <div class="logo">
                                            <a href="<?php print esc_url(home_url('/')); ?>">
                                                <img src="<?php echo esc_url($genix_logo); ?>" alt="<?php echo esc_attr($genix_alt); ?>">
                                            </a>
                                        </div>
                                    <?php endif; ?>
                                    <div class="tgmenu__navbar-wrap tgmenu__main-menu d-none d-lg-flex">
                                        <?php echo isset($menu_html) ? $menu_html : ''; ?>
                                    </div>

                                    <?php if (!empty($settings['header_search_show'] || $settings['header_offcanvas_show'] || $settings['tg_button_show'])) : ?>
                                        <div class="tgmenu__action d-none d-md-block">
                                            <ul class="list-wrap">
                                                <?php if (!empty($settings['header_search_show'])) : ?>
                                                    <li class="header-search">
                                                        <a href="javascript:void(0)" class="search-open-btn">
                                                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="none">
                                                                <path d="M19 19.0002L14.657 14.6572M14.657 14.6572C15.3999 13.9143 15.9892 13.0324 16.3912 12.0618C16.7933 11.0911 17.0002 10.0508 17.0002 9.00021C17.0002 7.9496 16.7933 6.90929 16.3913 5.93866C15.9892 4.96803 15.3999 4.08609 14.657 3.34321C13.9141 2.60032 13.0322 2.01103 12.0616 1.60898C11.0909 1.20693 10.0506 1 9.00002 1C7.94942 1 6.90911 1.20693 5.93848 1.60898C4.96785 2.01103 4.08591 2.60032 3.34302 3.34321C1.84269 4.84354 0.999817 6.87842 0.999817 9.00021C0.999817 11.122 1.84269 13.1569 3.34302 14.6572C4.84335 16.1575 6.87824 17.0004 9.00002 17.0004C11.1218 17.0004 13.1567 16.1575 14.657 14.6572Z" stroke="currentcolor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" />
                                                            </svg>
                                                        </a>
                                                    </li>
                                                <?php endif; ?>

                                                <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                                                    <li class="offCanvas-menu">
                                                        <a href="javascript:void(0)" class="menu-tigger">
                                                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                                <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                                <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                                <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                                <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                                <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                                <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                                <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                                <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                                <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                                            </svg>
                                                        </a>
                                                    </li>
                                                <?php endif; ?>

                                                <?php if (!empty($settings['tg_button_show'])) : ?>
                                                    <li class="header-btn">
                                                        <a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>>
                                                            <?php echo $settings['tg_btn_text']; ?>
                                                        </a>
                                                    </li>
                                                <?php endif; ?>
                                            </ul>
                                        </div>
                                    <?php endif; ?>

                                    <?php if (has_nav_menu('main-menu')) { ?>
                                        <div class="mobile-nav-toggler">
                                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 18 18" fill="none">
                                                <path d="M0 2C0 0.895431 0.895431 0 2 0C3.10457 0 4 0.895431 4 2C4 3.10457 3.10457 4 2 4C0.895431 4 0 3.10457 0 2Z" fill="currentcolor" />
                                                <path d="M0 9C0 7.89543 0.895431 7 2 7C3.10457 7 4 7.89543 4 9C4 10.1046 3.10457 11 2 11C0.895431 11 0 10.1046 0 9Z" fill="currentcolor" />
                                                <path d="M0 16C0 14.8954 0.895431 14 2 14C3.10457 14 4 14.8954 4 16C4 17.1046 3.10457 18 2 18C0.895431 18 0 17.1046 0 16Z" fill="currentcolor" />
                                                <path d="M7 2C7 0.895431 7.89543 0 9 0C10.1046 0 11 0.895431 11 2C11 3.10457 10.1046 4 9 4C7.89543 4 7 3.10457 7 2Z" fill="currentcolor" />
                                                <path d="M7 9C7 7.89543 7.89543 7 9 7C10.1046 7 11 7.89543 11 9C11 10.1046 10.1046 11 9 11C7.89543 11 7 10.1046 7 9Z" fill="currentcolor" />
                                                <path d="M7 16C7 14.8954 7.89543 14 9 14C10.1046 14 11 14.8954 11 16C11 17.1046 10.1046 18 9 18C7.89543 18 7 17.1046 7 16Z" fill="currentcolor" />
                                                <path d="M14 2C14 0.895431 14.8954 0 16 0C17.1046 0 18 0.895431 18 2C18 3.10457 17.1046 4 16 4C14.8954 4 14 3.10457 14 2Z" fill="currentcolor" />
                                                <path d="M14 9C14 7.89543 14.8954 7 16 7C17.1046 7 18 7.89543 18 9C18 10.1046 17.1046 11 16 11C14.8954 11 14 10.1046 14 9Z" fill="currentcolor" />
                                                <path d="M14 16C14 14.8954 14.8954 14 16 14C17.1046 14 18 14.8954 18 16C18 17.1046 17.1046 18 16 18C14.8954 18 14 17.1046 14 16Z" fill="currentcolor" />
                                            </svg>
                                        </div>
                                    <?php } ?>
                                </nav>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/mobile-menu.php'); ?>

            <?php if (!empty($settings['header_search_show'])) : ?>
                <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/header-search.php'); ?>
            <?php endif; ?>

            <?php if (!empty($settings['header_offcanvas_show'])) : ?>
                <?php include(GENIXCORE_ELEMENTS_PATH . '/header-info/offcanvas.php'); ?>
            <?php endif; ?>

        <?php endif; ?>

<?php
    }
}

$widgets_manager->register(new Genix_Header());
